//=============================================================================
// RPG Maker MZ - Button Picture (anti-spam edition)
//=============================================================================
/*:
 * @target MZ
 * @plugindesc Makes a picture clickable (with anti-spam: cooldown / dedupe / running-lock).
 * @author Yoji Ojima + tweak by HS
 *
 * @help ButtonPicture.js
 * ピクチャをボタン化し、クリックでコモンイベントを呼び出します。
 * 連打対策として「クールダウン」「重複予約禁止」「実行中ロック」を追加。
 *
 * 使い方：
 *  1) 「ピクチャの表示」でボタン画像を出す
 *  2) プラグインコマンド「ボタンピクチャの設定」で Picture番号 と コモンイベント を指定
 *  3) 連打対策はプラグインパラメータで調整
 *
 * @param cooldownMs
 * @text クリック連打無効化(ミリ秒)
 * @type number
 * @min 0
 * @default 800
 * @desc この時間内の再クリックを無効化します。0で無効（例：1000=1秒）
 *
 * @param lockWhileRunning
 * @text イベント実行中は無効
 * @type boolean
 * @default true
 * @desc マップ/コモンイベント実行中は新規クリックを受け付けません。
 *
 * @param dedupe
 * @text キュー重複禁止
 * @type boolean
 * @default true
 * @desc 同一コモンイベントがキュー内に既にある間は新規予約しません。
 *
 * @command set
 * @text ボタンピクチャの設定
 * @desc 指定したピクチャをクリック可能にします。
 *
 * @arg pictureId
 * @type number
 * @min 1
 * @max 100
 * @default 1
 * @text ピクチャ番号
 *
 * @arg commonEventId
 * @type common_event
 * @default 1
 * @text コモンイベント
 */

(() => {
  const pluginName = "ButtonPicture";
  const params = PluginManager.parameters(pluginName);
  const COOLDOWN_MS = Number(params.cooldownMs || 0);
  const LOCK_WHILE_RUNNING = String(params.lockWhileRunning) === "true";
  const DEDUPE = String(params.dedupe) === "true";

  const nowMs = () => (window.performance && performance.now) ? performance.now() : Date.now();
  const isCommonEventQueued = (id) => {
    const q = $gameTemp._commonEventQueue || [];
    return q.includes(id);
  };

  PluginManager.registerCommand(pluginName, "set", args => {
    const pictureId = Number(args.pictureId);
    const commonEventId = Number(args.commonEventId);
    const picture = $gameScreen.picture(pictureId);
    if (picture) picture.mzkp_commonEventId = commonEventId;
  });

  // クリック可能かの判定に、実行中ロック＆クールダウン＆重複抑止を追加
  Sprite_Picture.prototype.isClickEnabled = function () {
    const picture = this.picture();
    if (!picture || !picture.mzkp_commonEventId) return false;
    if ($gameMessage.isBusy()) return false;

    if (LOCK_WHILE_RUNNING) {
      // Mapイベント実行中 or コモンイベントが既に予約あり → 無効
      if ($gameMap.isEventRunning() || $gameTemp.isCommonEventReserved()) return false;
    }

    if (COOLDOWN_MS > 0) {
      const last = picture.mzkp_lastClickTime || 0;
      if (nowMs() - last < COOLDOWN_MS) return false;
    }

    if (DEDUPE && isCommonEventQueued(picture.mzkp_commonEventId)) {
      return false;
    }
    return true;
  };

  // クリック時の予約に、クールダウン開始＆重複予約ガードを追加
  Sprite_Picture.prototype.onClick = function () {
    const pic = this.picture();
    pic.mzkp_lastClickTime = nowMs();
    const id = pic.mzkp_commonEventId;
    if (!DEDUPE || !isCommonEventQueued(id)) {
      $gameTemp.reserveCommonEvent(id);
    }
  };

  // 既存の「ボタン押下中フラグ」判定はそのまま（UI挙動対策）
  Spriteset_Base.prototype.mzkp_isAnyPicturePressed = function () {
    return this._pictureContainer.children.some(sprite => sprite.isPressed());
  };

  const _Scene_Map_isAnyButtonPressed = Scene_Map.prototype.isAnyButtonPressed;
  Scene_Map.prototype.isAnyButtonPressed = function () {
    return (
      _Scene_Map_isAnyButtonPressed.apply(this, arguments) ||
      this._spriteset.mzkp_isAnyPicturePressed()
    );
  };
})();
